(* ::Subsubsection:: *)
(*KlevtsovTelescope package*)

(* ::Text:: *)
(*This impements the Klevtsov Telescope Combination optical system*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/KlevtsovTelescope"}]*)

BeginPackage["GeometricOptics`Packages`KlevtsovTelescope`", {"GeometricOptics`"}]

Options[KlevtsovTelescope] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
KlevtsovTelescope::usage="KlevtsovTelescope calculates parameters for a Maksutov Camera optical system.";

KlevtsovTelescopeInputPanel::usage = "KlevtsovTelescopeInputPanel gives the GUI panel to work with KlevtsovTelescope";

$KlevtsovTelescopeInputVariables = {"f1", "ft", "em", "diam", "\[Theta]"};

Begin["`Private`"]

KlevtsovTelescope[{f1_, ft_, em_, diam_, theta_}, opts___]:= KlevtsovTelescope[f1, ft, em, diam, theta, opts];

KlevtsovTelescope[f1_, ft_, em_, diam_, theta_, OptionsPattern[]] := 
 Quiet@Module[{em1, M, \[Beta], \[CapitalDelta], \[Gamma], R2, spc, c1, c2, c3, c4, rad, thick, ind, s3, s5, outputs, inPanel, outPanel},
 
		If[ArgumentsQ["KlevtsovTelescope", {$KlevtsovTelescopeInputVariables, {f1, ft, em, diam, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 9, "PackageName" -> "KlevtsovTelescope", "Arguments" -> <|"f1" -> f1, "ft" ->ft, "em" -> em, "diam" -> diam, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];

			em1 = (em + 40)/f1;
			M = ft/f1;
			\[Beta] = (M - em1)/(M + 1);
			\[CapitalDelta] = -\[Beta] f1;
			\[Gamma] = (2 M (1 + em1))/(1 - M^2);
			R2 = \[Gamma] f1;
			(* Corrector design *)
			(* First step *)
			spc = If[100 < diam <= 150, 25 , 30];
			rad = {-2 f1, 1/c4, 1/c3, 1/c2, R2, 1/c2, 1/c3, 1/c4};
			thick = {\[CapitalDelta], -spc, -3, 0, 3, 0, spc};
			ind = {{1, -1, -1.5187, -1, -1.5187, 1.5187, 1, 1.5187, 1}, {1, -1, -1.5224, -1, -1.5224, 1.5224, 1, 1.5224, 1}, {1, -1, -1.5143, -1, -1.5143, 1.5143, 1, 1.5143, 1}};
			TotalAberrations[rad, thick, ind, {0, 0, 0, 0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, y1, theta, {0.54, \[Lambda]2, \[Lambda]3}];
			s3 = FindRoot[{GOfocalLength[[1]] == ft, GOaberration[[1]] == 0, GOcoma[[1]] == 0}, {c2, 1/R2}, {c3, 0.007}, {c4, 0.007}, MaxIterations -> 300];
			
			(* Second step *)
			rad = {1/c1, 1/c4, 1/c3, 1/c2, R2, 1/c2, 1/c3, 1/c4};
			thick = {\[CapitalDelta], -spc, 0, -3, 3, 0, spc};
			TotalAberrations[rad, thick, ind, {0, 0, 0, 0, 0, 0, 0, 0}, diam/2, 0, 0, -Infinity, y1, theta, {\[Lambda]1, \[Lambda]2, \[Lambda]3}];
			
			s5 = FindRoot[{GOfocalLength[[1]] == ft, GOaberration[[1]] == 0, GOcoma[[1]] == 0, GOdistancegauss[[2, GOn]] - GOdistancegauss[[3, GOn]] == 0}, 
							{c1, -1/(2 f1)}, {c2, ReplaceAll[c2, s3]}, {c3, ReplaceAll[c3, s3]}, {c4, ReplaceAll[c4, s3]}];

			outputs = {{"Primary radius", N[1/ReplaceAll[c1, s5]]},
						{"Secondary radius", R2 // N},
						{"Distance between primary and secondary", N[\[CapitalDelta]]},
						{"Back distance", (GOdistancegauss[[1, GOn]] /. s5) + \[CapitalDelta]},
						{"Obstruction factor", N[1 - \[Beta]]},
						{"Meniscus thickness", spc},
						{"Radius of mangin's mirror", N[1/ReplaceAll[c2, s5]]},
						{"Radius of the meniscus surface in contact with the Mangin mirror", 1/ReplaceAll[c3, s5]},
						{"Radius of the second surface of the meniscus ", 1/ReplaceAll[c4, s5]}, 
						{"Total focal length", GOfocalLength[[1]] /. s5},
						{"Height of the image", GOimageHeight[[1, GOn]] /. s5},
						{"Green back foc.", GOdistancegauss[[1, GOn]] /. s5},
						{"Bleu back foc.", GOdistancegauss[[2, GOn]] /. s5},
						{"Red back foc.", GOdistancegauss[[3, GOn]] /. s5},
						{"Difference between bleu and red back foc.", (GOdistancegauss[[2, GOn]] /. s5) - (GOdistancegauss[[3, GOn]] /. s5)},
						{"Aberration", GOaberration[[1]] /. s5},
						{"Coma", GOcoma[[1]] /. s5},
						{"Astigmatism", GOastigmatism[[1]] /. s5},
						{"Curvature", GOcurvature[[1]] /. s5}};

			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"Total focal length", "ft", ft},
							{"Back distance", "em", em},
							{"Diameter of primary mirror", "diam", diam},
							{"Field angle  in degree", "theta", theta}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];
				
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "KlevtsovTelescope" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "KlevtsovTelescope", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "KlevtsovTelescope" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "KlevtsovTelescope", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1" , f1},
														 {"ft", ft},
														 {"em", em},
														 {"diam", diam},
														 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["KlevtsovTelescope not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

KlevtsovTelescopeInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "KlevtsovTelescope"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, ft, em, diam, theta} = ReplaceAll[$KlevtsovTelescopeInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run "<>package<>" function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the value of focal length of the primary mirror"]},
						  {"Total focal length" , "ft", Tooltip[InputField[Dynamic[ft], Alignment -> Center], "Insert the value of total focal length"]},
						  {"Back distance" , "em", Tooltip[InputField[Dynamic[em], Alignment -> Center], "Insert the value of back distance"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, ft, em, diam, theta}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, ft, em, diam, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]